(() => {
  const root = document.documentElement;
  const themeToggle = document.querySelector('[data-theme-toggle]');
  const localeSwitch = document.querySelector('[data-locale-switch]');
  const currencySwitch = document.querySelector('[data-currency-switch]');
  const aiPrompt = document.querySelector('[data-ai-prompt]');
  const aiGenerate = document.querySelector('[data-ai-generate]');
  const aiRegenerate = document.querySelector('[data-ai-regenerate]');
  const aiResults = document.querySelector('[data-ai-results]');
  const chatContainer = document.querySelector('[data-quanta-chat]');
  const chatMessages = document.querySelector('[data-chat-messages]');
  const chatInput = document.querySelector('[data-chat-input]');
  const chatSend = document.querySelector('[data-chat-send]');
  const chatModel = document.querySelector('[data-chat-model]');
  const chatList = document.querySelector('[data-chat-list]');
  const chatNew = document.querySelector('[data-new-chat]');
  const memoryToggle = document.querySelector('[data-memory-toggle]');
  const planSelector = document.querySelector('[data-plan-set]');
  const chatMore = document.querySelector('[data-chat-more]');
  const chatMoreMenu = document.querySelector('[data-chat-more-menu]');
  const chatMoreClose = document.querySelector('[data-chat-more-close]');
  const chatImage = document.querySelector('[data-chat-image]');
  const chatDoc = document.querySelector('[data-chat-doc]');
  let pendingImage = null;
  let pendingDoc = null;
  const mobileMenuToggle = document.querySelector('[data-menu-toggle]');
  const mobileNav = document.querySelector('[data-main-nav-mobile]');
  const modalOverlay = document.querySelector('[data-modal-overlay]');
  const settingsModal = document.querySelector('[data-settings-modal]');
  const openSettings = document.querySelectorAll('[data-open-settings]');
  const closeSettings = document.querySelectorAll('[data-close-settings]');
  const cartDrawer = document.querySelector('[data-cart-drawer]');
  const accountDrawer = document.querySelector('[data-account-drawer]');
  const openCart = document.querySelectorAll('[data-open-cart]');
  const openAccount = document.querySelectorAll('[data-open-account]');
  const closeDrawer = document.querySelectorAll('[data-close-drawer]');
  let currentChatId = 'default';
  const stored = localStorage.getItem('theme');
  const initial = root.dataset.theme || 'auto';

  const applyTheme = (theme) => {
    if (theme === 'auto') {
      root.dataset.theme = window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light';
    } else {
      root.dataset.theme = theme;
    }
    localStorage.setItem('theme', theme);
  };

  applyTheme(stored || initial);

  if (themeToggle) {
    themeToggle.addEventListener('click', () => {
      const current = localStorage.getItem('theme') || initial || 'auto';
      const next = current === 'light' ? 'dark' : current === 'dark' ? 'auto' : 'light';
      applyTheme(next);
      themeToggle.innerText = `Theme: ${root.dataset.theme}`;
    });
    themeToggle.innerText = `Theme: ${root.dataset.theme}`;
  }

  const postJson = async (url, body) => {
    try {
      const res = await fetch(url, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams(body),
      });
      if (!res.ok) throw new Error('Request failed');
      return res.json();
    } catch (err) {
      console.warn(err);
      return null;
    }
  };

  if (localeSwitch) {
    localeSwitch.addEventListener('change', async (e) => {
      const value = e.target.value;
      const result = await postJson('/locale', { locale: value });
      if (result?.status === 'ok') {
        window.location.reload();
      }
    });
  }

  if (currencySwitch) {
    currencySwitch.addEventListener('change', async (e) => {
      const value = e.target.value;
      const result = await postJson('/currency', { currency: value });
      if (result?.status === 'ok') {
        window.location.reload();
      }
    });
  }

  const renderSuggestions = (text) => {
    if (!aiResults) return;
    aiResults.innerHTML = '';
    if (!text) {
      aiResults.innerHTML = '<p class="muted">No suggestions yet.</p>';
      return;
    }
    const pre = document.createElement('pre');
    pre.textContent = text;
    aiResults.appendChild(pre);
  };

  const fetchIdeas = async () => {
    if (!aiPrompt || !aiResults) return;
    const prompt = aiPrompt.value.trim();
    if (!prompt) {
      renderSuggestions('Enter a brand or idea first.');
      return;
    }
    aiResults.innerHTML = '<p class="muted">Loading...</p>';
    const res = await postJson('/ai/domain-ideas', { prompt });
    if (res?.status === 'ok') {
      renderSuggestions(res.suggestions);
    } else {
      renderSuggestions(res?.message || 'Error generating ideas.');
    }
  };

  if (aiGenerate) {
    aiGenerate.addEventListener('click', fetchIdeas);
  }
  if (aiRegenerate) {
    aiRegenerate.addEventListener('click', fetchIdeas);
  }

  const appendMessage = (role, text) => {
    if (!chatMessages) return;
    const bubble = document.createElement('div');
    bubble.className = 'glass card';
    bubble.innerHTML = `<strong>${role}</strong><p>${text}</p>`;
    chatMessages.appendChild(bubble);
    chatMessages.scrollTop = chatMessages.scrollHeight;
  };

  const sendChat = async () => {
    if (!chatInput || !chatMessages) return;
    const text = chatInput.value.trim();
    if (!text) return;
    appendMessage('You', text);
    chatInput.value = '';
    const form = new FormData();
    form.append('message', text);
    form.append('model', chatModel ? chatModel.value : 'quanta-1');
    form.append('chat_id', currentChatId);
    if (pendingImage) {
      form.append('image', pendingImage);
    }
    if (pendingDoc) {
      form.append('doc', pendingDoc);
    }
    const res = await fetch('/ai/quanta-chat', {
      method: 'POST',
      body: form,
    }).then(r => r.json()).catch(() => null);
    if (res?.status === 'ok') {
      appendMessage('Quanta', res.reply || '');
    } else {
      appendMessage('System', res?.message || 'Error sending message');
    }
    pendingImage = null;
    pendingDoc = null;
    toggleMore(false);
  };

  if (chatSend) {
    chatSend.addEventListener('click', sendChat);
  }
  if (chatInput) {
    chatInput.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' && !e.shiftKey) {
        e.preventDefault();
        sendChat();
      }
    });
  }

  if (chatList) {
    chatList.addEventListener('click', (e) => {
      const btn = e.target.closest('[data-chat-id]');
      if (!btn) return;
      currentChatId = btn.getAttribute('data-chat-id');
      chatMessages.innerHTML = '<p class="muted">Switched to chat: ' + currentChatId + '</p>';
    });
  }

  const loadHistory = async () => {
    const res = await postJson('/ai/quanta-history', {});
    if (res?.status === 'ok' && chatList) {
      chatList.innerHTML = '';
      res.history.forEach((item, idx) => {
        const li = document.createElement('li');
        li.innerHTML = `<button class="nav-link" data-chat-id="${item.chat_id}" ${idx === 0 ? 'aria-current="true"' : ''}>${item.chat_id} (${item.count})</button>`;
        chatList.appendChild(li);
      });
      if (res.history[0]) {
        currentChatId = res.history[0].chat_id;
      }
    }
  };

  if (chatNew) {
    chatNew.addEventListener('click', async () => {
      const res = await postJson('/ai/quanta-create', {});
      if (res?.status === 'ok' && chatList) {
        const li = document.createElement('li');
        li.innerHTML = `<button class="nav-link" data-chat-id="${res.chat_id}">${res.chat_id}</button>`;
        chatList.appendChild(li);
        currentChatId = res.chat_id;
      }
    });
  }

  if (memoryToggle) {
    memoryToggle.addEventListener('click', async () => {
      const enabled = memoryToggle.getAttribute('data-enabled') !== 'true';
      const res = await postJson('/ai/memory-toggle', { enabled });
      if (res?.status === 'ok') {
        memoryToggle.setAttribute('data-enabled', String(res.memory_enabled));
        memoryToggle.innerText = res.memory_enabled ? 'Memory: On' : 'Memory: Off';
      }
    });
  }

  if (planSelector) {
    planSelector.addEventListener('change', async (e) => {
      const plan = e.target.value;
      await postJson('/plan/set', { plan });
    });
  }

  loadHistory();

  const toggleMore = (show) => {
    if (!chatMoreMenu) return;
    chatMoreMenu.style.display = show ? 'block' : 'none';
  };

  if (chatMore) {
    chatMore.addEventListener('click', () => toggleMore(chatMoreMenu.style.display !== 'block'));
  }
  if (chatMoreClose) {
    chatMoreClose.addEventListener('click', () => toggleMore(false));
  }

  const handleAttachment = (file, label) => {
    if (!file) return;
    appendMessage('Attachment', `${label}: ${file.name}`);
    if (label === 'Image') pendingImage = file;
    if (label === 'Document') pendingDoc = file;
  };

  if (chatImage) {
    chatImage.addEventListener('change', (e) => {
      handleAttachment(e.target.files[0], 'Image');
      e.target.value = '';
      toggleMore(false);
    });
  }
  if (chatDoc) {
    chatDoc.addEventListener('change', (e) => {
      handleAttachment(e.target.files[0], 'Document');
      e.target.value = '';
      toggleMore(false);
    });
  }

  if (mobileMenuToggle && mobileNav) {
    mobileMenuToggle.addEventListener('click', () => {
      mobileNav.classList.toggle('open');
    });
  }

  const showModal = () => {
    if (modalOverlay) modalOverlay.style.display = 'block';
    if (settingsModal) settingsModal.style.display = 'block';
  };
  const hideModal = () => {
    if (modalOverlay) modalOverlay.style.display = 'none';
    if (settingsModal) settingsModal.style.display = 'none';
  };
  openSettings.forEach(btn => btn.addEventListener('click', showModal));
  closeSettings.forEach(btn => btn.addEventListener('click', hideModal));
  if (modalOverlay) modalOverlay.addEventListener('click', hideModal);

  const openDrawer = (drawer) => {
    if (drawer) drawer.classList.add('open');
    if (modalOverlay) modalOverlay.style.display = 'block';
  };
  const closeAllDrawers = () => {
    [cartDrawer, accountDrawer].forEach(d => d && d.classList.remove('open'));
    if (modalOverlay) modalOverlay.style.display = 'none';
  };
  openCart.forEach(btn => btn.addEventListener('click', () => openDrawer(cartDrawer)));
  openAccount.forEach(btn => btn.addEventListener('click', () => openDrawer(accountDrawer)));
  closeDrawer.forEach(btn => btn.addEventListener('click', closeAllDrawers));
  if (modalOverlay) modalOverlay.addEventListener('click', closeAllDrawers);
})();
