<?php

namespace App\Services;

class HostingService
{
    private string $whmHost;
    private string $whmToken;

    public function __construct()
    {
        $this->whmHost = getenv(config('whm_api_host_env', 'WHM_API_HOST')) ?: '';
        $this->whmToken = getenv(config('whm_api_token_env', 'WHM_API_TOKEN')) ?: '';
    }

    public function createAccount(string $domain, string $plan): array
    {
        if (!$this->isConfigured()) {
            return $this->stub("Create account for {$domain} on plan {$plan}");
        }
        return $this->whmCall('createacct', [
            'username' => substr(preg_replace('/[^a-z0-9]/', '', $domain), 0, 8) ?: 'user' . rand(100, 999),
            'domain' => $domain,
            'plan' => $plan,
        ]);
    }

    public function suspendAccount(string $domain): array
    {
        if (!$this->isConfigured()) {
            return $this->stub("Suspend account {$domain}");
        }
        return $this->whmCall('suspendacct', ['domain' => $domain]);
    }

    public function terminateAccount(string $domain): array
    {
        if (!$this->isConfigured()) {
            return $this->stub("Terminate account {$domain}");
        }
        return $this->whmCall('removeacct', ['domain' => $domain]);
    }

    public function resetPassword(string $domain): array
    {
        if (!$this->isConfigured()) {
            return $this->stub("Reset password for {$domain}");
        }
        return $this->whmCall('passwd', ['domain' => $domain, 'password' => bin2hex(random_bytes(5))]);
    }

    private function whmCall(string $function, array $params): array
    {
        $url = rtrim($this->whmHost, '/') . "/json-api/{$function}?api.version=1";
        if (!empty($params)) {
            $url .= '&' . http_build_query($params);
        }
        $headers = ["Authorization: whm root:" . $this->whmToken];

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_SSL_VERIFYPEER => false,
        ]);
        $raw = curl_exec($ch);
        $err = curl_error($ch);
        $status = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
        curl_close($ch);

        if ($err) {
            return ['status' => 'error', 'message' => $err];
        }
        if ($status >= 400) {
            return ['status' => 'error', 'message' => 'WHM HTTP ' . $status, 'raw' => $raw];
        }
        $json = json_decode($raw, true);
        return $json ?: ['status' => 'ok', 'raw' => $raw];
    }

    private function isConfigured(): bool
    {
        return $this->whmHost && $this->whmToken;
    }

    private function stub(string $message): array
    {
        return ['status' => 'queued', 'message' => $message, 'note' => 'WHM not configured; stubbed'];
    }
}
