<?php

namespace App\Services;

class GeminiClient
{
    private string $apiKey;
    private string $endpoint;

    public function __construct()
    {
        $envKey = config('gemini_api_key_env', 'GEMINI_BACKEND_API_KEY');
        $this->apiKey = getenv($envKey) ?: '';
        $this->endpoint = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-3-flash-preview:generateContent';
    }

    public function isConfigured(): bool
    {
        return !empty($this->apiKey);
    }

    public function generate(array $messages): array
    {
        if (!$this->isConfigured()) {
            return ['error' => 'API key missing'];
        }

        $payload = [
            'contents' => array_map(function ($m) {
                return ['parts' => [['text' => $m['text'] ?? '']], 'role' => $m['role'] ?? 'user'];
            }, $messages),
            'safetySettings' => [],
        ];

        $url = $this->endpoint . '?key=' . urlencode($this->apiKey);
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_TIMEOUT => 15,
        ]);
        $raw = curl_exec($ch);
        $err = curl_error($ch);
        $status = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
        curl_close($ch);

        if ($err) {
            return ['error' => 'Request error: ' . $err];
        }
        if ($status >= 400) {
            return ['error' => 'Gemini returned HTTP ' . $status, 'raw' => $raw];
        }

        $data = json_decode($raw, true);
        if (!$data) {
            return ['error' => 'Invalid response'];
        }
        return ['data' => $data];
    }
}
