<?php

namespace App\Core;

use App\Controllers\ErrorController;

class Router
{
    private array $routes = [];
    private $notFoundHandler = null;

    public function get(string $path, callable|array $action): void
    {
        $this->addRoute('GET', $path, $action);
    }

    public function post(string $path, callable|array $action): void
    {
        $this->addRoute('POST', $path, $action);
    }

    public function any(string $path, callable|array $action): void
    {
        foreach (['GET', 'POST', 'PUT', 'PATCH', 'DELETE'] as $verb) {
            $this->addRoute($verb, $path, $action);
        }
    }

    public function setNotFound(callable|array $action): void
    {
        $this->notFoundHandler = $action;
    }

    private function addRoute(string $method, string $path, callable|array $action): void
    {
        $pattern = $this->compilePath($path);
        $this->routes[] = compact('method', 'pattern', 'action', 'path');
    }

    public function dispatch(string $uri, string $method): mixed
    {
        $path = parse_url($uri, PHP_URL_PATH) ?: '/';
        $path = rtrim($path, '/') ?: '/';
        foreach ($this->routes as $route) {
            if ($route['method'] !== $method) {
                continue;
            }

            $matches = [];
            if (preg_match($route['pattern'], $path, $matches)) {
                array_shift($matches);
                return $this->invoke($route['action'], $matches);
            }
        }

        http_response_code(404);
        if ($this->notFoundHandler) {
            return $this->invoke($this->notFoundHandler, [$path]);
        }
        return $this->renderNotFound($path);
    }

    private function invoke(callable|array $action, array $params): mixed
    {
        if (is_array($action)) {
            [$class, $method] = $action;
            $controller = new $class();
            return $controller->$method(...$params);
        }

        return $action(...$params);
    }

    private function compilePath(string $path): string
    {
        $escaped = preg_replace('#\{([^}/]+)\}#', '([^/]+)', $path);
        // Allow optional trailing slash
        if (!str_ends_with($escaped, '/')) {
            $escaped .= '/?';
        }
        return '#^' . $escaped . '$#';
    }

    private function renderNotFound(string $path): string
    {
        if (class_exists(ErrorController::class)) {
            $controller = new ErrorController();
            return $controller->notFound($path);
        }
        return "<h1>404</h1><p>Page not found: {$path}</p>";
    }
}
