<?php

namespace App\Controllers;

use App\Services\GeminiClient;

class AiController extends BaseController
{
    public function domainIdeas(): string
    {
        $prompt = trim($_POST['prompt'] ?? '');
        if ($prompt === '') {
            http_response_code(400);
            return json_encode(['status' => 'error', 'message' => 'Prompt required']);
        }

        $client = new GeminiClient();
        if (!$client->isConfigured()) {
            http_response_code(500);
            return json_encode(['status' => 'error', 'message' => 'Gemini not configured']);
        }

        $messages = [
            [
                'role' => 'system',
                'text' => 'You suggest brandable, short, easy-to-pronounce domain names with diverse TLDs. Return JSON list of objects: {domain, tld, rationale}. Max 10 suggestions.',
            ],
            [
                'role' => 'user',
                'text' => $prompt,
            ],
        ];

        $result = $client->generate($messages);
        if (isset($result['error'])) {
            http_response_code(500);
            return json_encode(['status' => 'error', 'message' => $result['error']]);
        }

        $text = $this->extractText($result['data'] ?? []);
        return json_encode(['status' => 'ok', 'suggestions' => $text]);
    }

    public function quantaChat(): string
    {
        $message = trim($_POST['message'] ?? '');
        $model = $_POST['model'] ?? 'quanta-1';
        $chatId = $_POST['chat_id'] ?? 'default';
        $memoryEnabled = $_SESSION['memory_enabled'] ?? true;
        $plan = $_SESSION['plan'] ?? 'free';
        $proAllowed = in_array($plan, ['pro', 'business'], true);
        if ($model === 'quanta-2' && !$proAllowed) {
            http_response_code(403);
            return json_encode(['status' => 'error', 'message' => 'Quanta 2 is available for Pro/Business']);
        }
        if ($message === '') {
            http_response_code(400);
            return json_encode(['status' => 'error', 'message' => 'Message required']);
        }

        $client = new GeminiClient();
        if (!$client->isConfigured()) {
            http_response_code(500);
            return json_encode(['status' => 'error', 'message' => 'Gemini not configured']);
        }

        $system = 'You are Quanta Chat, a concise assistant. Keep responses focused and helpful.';
        if ($model === 'quanta-2') {
            $system .= ' Use more advanced reasoning when beneficial.';
        }

        $attachmentContext = $this->attachmentContext();

        $history = $_SESSION['quanta_history'][$chatId] ?? [];
        $messages = array_merge($history, [
            ['role' => 'system', 'text' => $system],
            ['role' => 'system', 'text' => $attachmentContext],
            ['role' => 'user', 'text' => $message],
        ]);

        $result = $client->generate($messages);
        if (isset($result['error'])) {
            http_response_code(500);
            return json_encode(['status' => 'error', 'message' => $result['error']]);
        }

        $text = $this->extractText($result['data'] ?? []);
        $messages[] = ['role' => 'assistant', 'text' => $text];
        $_SESSION['quanta_history'][$chatId] = $messages;

        if ($memoryEnabled) {
            $_SESSION['quanta_memory'][$chatId][] = $this->summarizeMemory($message, $text);
        }

        return json_encode(['status' => 'ok', 'reply' => $text, 'chat_id' => $chatId, 'memory' => $memoryEnabled]);
    }

    private function extractText(array $data): string
    {
        $candidates = $data['candidates'][0]['content']['parts'][0]['text'] ?? '';
        return is_string($candidates) ? $candidates : '';
    }

    private function summarizeMemory(string $user, string $assistant): string
    {
        $joined = substr($user . ' | ' . $assistant, 0, 300);
        return $joined;
    }

    public function quantaHistory(): string
    {
        $history = $_SESSION['quanta_history'] ?? [];
        $summaries = [];
        foreach ($history as $id => $messages) {
            $last = end($messages);
            $summaries[] = [
                'chat_id' => $id,
                'last' => is_array($last) && isset($last['text']) ? substr($last['text'], 0, 80) : '',
                'count' => count($messages),
            ];
        }
        return json_encode(['status' => 'ok', 'history' => $summaries]);
    }

    public function quantaCreate(): string
    {
        $chatId = 'chat_' . bin2hex(random_bytes(3));
        $_SESSION['quanta_history'][$chatId] = [];
        return json_encode(['status' => 'ok', 'chat_id' => $chatId]);
    }

    public function quantaClear(): string
    {
        $chatId = $_POST['chat_id'] ?? '';
        if (!$chatId) {
            http_response_code(400);
            return json_encode(['status' => 'error', 'message' => 'Chat id required']);
        }
        unset($_SESSION['quanta_history'][$chatId], $_SESSION['quanta_memory'][$chatId]);
        return json_encode(['status' => 'ok', 'chat_id' => $chatId]);
    }

    public function memoryToggle(): string
    {
        $enabled = filter_var($_POST['enabled'] ?? true, FILTER_VALIDATE_BOOLEAN);
        $_SESSION['memory_enabled'] = $enabled;
        if (!$enabled) {
            $_SESSION['quanta_memory'] = [];
        }
        return json_encode(['status' => 'ok', 'memory_enabled' => $enabled]);
    }

    private function attachmentContext(): string
    {
        $parts = [];
        if (!empty($_FILES['doc']) && is_uploaded_file($_FILES['doc']['tmp_name'])) {
            $content = '';
            if ($_FILES['doc']['size'] < 1024 * 1024) {
                $raw = file_get_contents($_FILES['doc']['tmp_name']);
                $content = substr($raw ?: '', 0, 2000);
            }
            $parts[] = 'Document attached: ' . ($_FILES['doc']['name'] ?? '') . ' Content snippet: ' . $content;
        }
        if (!empty($_FILES['image']) && is_uploaded_file($_FILES['image']['tmp_name'])) {
            $parts[] = 'Image attached: ' . ($_FILES['image']['name'] ?? '');
        }
        return $parts ? implode("\n", $parts) : 'No attachments provided.';
    }
}
